
downloadedPackages <- as.data.frame(installed.packages())$Package

requiredPackages <- c("mpi", # for accessing data
                      "tidyverse", # for manipulating data
                      "data.table", # for manipulating data FAST
                      "ggrba", # for graphing
                      "zoo",  # for working with dates
                      "readxl",
                      "arphit",
                      "bankRun",
                      "ivreg",
                      "zoo",
                      "dynlm",
                      "rfido",
                      "plm")

if(any(!requiredPackages %in% downloadedPackages)){
  
  install.packages(requiredPackages[!requiredPackages %in% downloadedPackages])
  
}

for(i in requiredPackages){library(i, character.only = TRUE)}

projectDir <- "insert project dir"

# Lookup tables -----------------------------------------------------------

# Top-level LCR lookup (NCOs, HQLA, etc) 

lookupLCRForm <- readxl::read_excel("not included in supp info")

# Lookup for LCR weights by category 

LCRweightsLookup <- readxl::read_excel("not included in supp info", 
                                       sheet = 1) 

liabilitiesWaterfallLookup <- LCRweightsLookup %>%
  select(-c(LCRamount, LCRweightedAmount, AssetOrLiabilities, DomOff)) %>% 
  distinct() %>% 
  pivot_longer(-c(Measure))

additionalCat <- readxl::read_excel("not included in supp info", 
                                    sheet = 2)

LCRLookupByComponent <- LCRweightsLookup %>%
  select(Measure, LCRamount, LCRweightedAmount) %>% 
  filter(!LCRamount %in% additionalCat$Measure) %>% 
  bind_rows(additionalCat) %>% 
  distinct() %>% 
  pivot_longer(-Measure) %>% 
  mutate(name = gsub("LCR", "", name))

# Lookup for assets and liabilities lookup 
maturityBucketsLookup <- readxl::read_excel("not included in supp info") %>% 
  pivot_longer(-c(Measure, AssetOrLiabilities), names_to = "tenor", values_to = "value")

offBalanceSheetLookup <- readxl::read_excel("not included in supp info", 
                                            sheet = "OffBalanceSheet")

offBalanceSheetAttrs <- offBalanceSheetLookup$attributeName

# Asset waterfall lookup (inflows)

inflow <- readxl::read_excel("not included in supp info", 
                             sheet = "Inflows Lookup")

attrLookupInflows <- inflow %>% 
  pivot_longer(-c(measure, Group))

inflowFormItems <- attrLookupInflows$value

# LCR inflow lookup
inflowLookupTable <- readxl::read_excel("not included in supp info", 
                                        sheet = "Inflows")

inflowAttrs <- inflowLookupTable$Weighted
inflowAttrsUnweighted <- inflowLookupTable$Unweighted

fidoLCRAttr <- unique(c(LCRweightsLookup$LCRamount, LCRweightsLookup$LCRweightedAmount)) %>% 
  .[! . %in% c(additionalCat$Measure)] %>% 
  c(., additionalCat$LCRamount, additionalCat$LCRweightedAmount)

fidoLiabilitiesTermAttr <- unique(c(LCRweightsLookup$Overnight,
                                    LCRweightsLookup$`2 to 7 days`, 
                                    LCRweightsLookup$`8 to 15 days`, 
                                    LCRweightsLookup$`16 to 30 days`))

LCRbanksTestList <- c("Commonwealth Bank of Australia",
                      "Westpac Banking Corporation",
                      "Australia and New Zealand Banking Group Limited",
                      "National Australia Bank Limited",
                      "AMP Bank Limited",
                      "Bendigo and Adelaide Bank Limited",
                      "Bank of Queensland Limited",
                      "Macquarie Bank Limited",
                      "Suncorp-Metway Limited",
                      "Bank of China (Australia) Limited",
                      "HSBC Bank Australia Limited",
                      "ING Bank (Australia) Limited",
                      "Rabobank Australia Limited")


# Data pull ---------------------------------------------------------------

## Liability waterfalls ---------------------------------------------------------------

# These data given banks' liabilities by when they come due
liabilitiesTermData <- fido() %>%
  filter(source_form_code %in% c("ARF_210_3_2"), 
         attribute_name %in% fidoLiabilitiesTermAttr, 
         entity_name %in% LCRbanksTestList,
         highest_consolidation_indicator == "Y",
         date >= as.Date("2018-01-01")) %>%   
  select(source_form_code, 
         rba_entity_shortname,
         entity_name,
         attribute_name, 
         entity_level,
         date, 
         value_number) %>% 
  as_tibble()

lookup <- liabilitiesTermData %>% 
  distinct(entity_name, rba_entity_shortname) %>% 
  group_by(rba_entity_shortname) %>% 
  slice(1) %>% 
  ungroup()

## Asset waterfalls ---------------------------------------------------------

inflowDataRaw <- fido() %>%
  filter(source_form_code %in% c("ARF_210_3_1"), 
         entity_name %in% LCRbanksTestList, 
         attribute_name %in% inflowFormItems, # total AGS  
         highest_consolidation_indicator == "Y",
         date >= as.Date("2018-01-01")) %>%   
  select(source_form_code, 
         rba_entity_shortname,
         entity_name,
         attribute_name, 
         entity_level,
         date, 
         value_number) %>% 
  as_tibble()

## LCR Form - totals ---------------------------------------------------------------

# Gives HQLA, NCOs, ES balances etc. 
level2 <- bankRun::sql_run("not included in supp info", #Level 2
                           "FIDO_Prod", 
                           override = TRUE) %>% 
  dplyr::mutate(level = 2)

level1 <- bankRun::sql_run("not included in supp info", #Level 1
                           "FIDO_Prod", 
                           override = TRUE) %>% 
  dplyr::mutate(level = 1)

nonBanks <- bankRun::sql_run("not included in supp info", #Level 1
                             "FIDO_Prod", 
                             override = TRUE) %>% 
  dplyr::mutate(level = 3) %>% 
  dplyr::filter(!institution_code %in% c(level1$institution_code, level2$institution_code))

HQLAData <- dplyr::bind_rows(level1, level2) %>% 
  dplyr::filter(Attribute_Name %in% lookupLCRForm$attributeName) %>% 
  dplyr::group_by(Attribute_Name, institution_code, period) %>% 
  dplyr::arrange(institution_code, Attribute_Name, period, level) %>% 
  dplyr::mutate(count = dplyr::n()) %>%   
  dplyr::ungroup() %>% 
  dplyr::filter(count == 1 | (count == 2 & level == 2)) %>%
  dplyr::bind_rows(nonBanks) %>% 
  dplyr::distinct() %>% 
  dplyr::filter(name %in% LCRbanksTestList) %>% # !!!
  dplyr::left_join(lookupLCRForm, by = c("Attribute_Name" = "attributeName")) %>% 
  dplyr::select(period,level,Label, institution_code, name,value) %>% 
  tidyr::drop_na() %>% 
  tidyr::pivot_wider(names_from = Label, values_from = value) %>% 
  dplyr::select(period, 
                level, 
                institution_code, 
                name, 
                Notes, 
                ESBalances, 
                OtherCBBalances, 
                AGS, 
                Semis,
                `Total HQLA`,
                `Alt HQLA`, 
                `Total cash inflows`, 
                `Total cash inflows after inflow cap`,
                `Outflows`, 
                `Net Cash Outflow`, 
                LCR, 
                `Mean LCR during reporting period`) %>% 
  dplyr::mutate(LCRcalc = (`Total HQLA` + `Alt HQLA`)/`Net Cash Outflow`*100, 
                diff = LCR - LCRcalc) %>% 
  dplyr::arrange(institution_code, 
                 period)

## LCR form - by components -----------------------------------------------------

LCRweightsData <- fido() %>%
  filter(source_form_code %in% c("ARF_210_1A"), 
         attribute_name %in% fidoLCRAttr, 
         entity_name %in% LCRbanksTestList,
         highest_consolidation_indicator == "Y",
         date >= as.Date("2018-01-01")) %>%   
  select(source_form_code, 
         rba_entity_shortname,
         entity_name,
         attribute_name, 
         entity_level,
         date, 
         value_number) %>% 
  as_tibble()

## LCR Form - off-balance sheet liabilities --------------------------------

outflowsOffBalanceSheetRaw <- fido() %>%
  filter(source_form_code %in% c("ARF_210_1A"), 
         attribute_name %in% offBalanceSheetAttrs, 
         entity_name %in% LCRbanksTestList,
         highest_consolidation_indicator == "Y",
         date >= as.Date("2015-01-01")) %>%   
  select(source_form_code, 
         rba_entity_shortname,
         entity_name,
         attribute_name, 
         entity_level,
         date, 
         value_number) %>% 
  as_tibble()

## LCR form - inflows ------------------------------------------------------

inflowComponentsFromLCR <- fido() %>%
  filter(source_form_code %in% c("ARF_210_1A"), 
         attribute_name %in% c(inflowAttrs,inflowAttrsUnweighted), # total AGS  
         entity_name %in% LCRbanksTestList,
         highest_consolidation_indicator == "Y",
         date >= as.Date("2018-01-01")) %>%   
  select(source_form_code, 
         rba_entity_shortname,
         entity_name,
         attribute_name, 
         entity_level,
         date, 
         value_number) %>% 
  as_tibble()

# Total Liabilities and Capital -------------------------------------------

totalLiabilitiesAndCapital <- fido() %>%
  filter(source_form_code %in% c("ARF_210_3_2"), 
         attribute_name %in% c("E15773"), 
         entity_name %in% LCRbanksTestList,
         highest_consolidation_indicator == "Y",
         date >= as.Date("2016-01-01")) %>%   
  select(source_form_code, 
         rba_entity_shortname,
         entity_name,
         attribute_name, 
         entity_level,
         date, 
         value_number) %>% 
  as_tibble()

# Calculations ------------------------------------------------------

## Outflows ----------------------------------------------------------------

# NCOs by category over time. Pull in weighted and unweighted 
# values to calculate an average weight 

LCROutflowsWeighted <- LCRweightsData %>% 
  group_by(date, attribute_name) %>% 
  summarise(total = sum(value_number)) %>% 
  ungroup() %>%
  left_join(LCRLookupByComponent %>% 
              select(Measure, name, value), 
            by = c("attribute_name" = "value")) %>% 
  select(-attribute_name) %>%
  group_by(date, Measure, name) %>% # this line of code combines all the other categories to calculate a common weighting to be applied to the "other" category in the liabilities form. 
  summarise(total = sum(total)/1e9) %>% 
  ungroup() %>% 
  pivot_wider(names_from = "name", values_from = "total") %>% 
  mutate(averageWeight = weightedAmount/amount) %>% 
  select(date, Measure, amount, weightedAmount, averageWeight)

# Liabilities by category and residual maturity over time
outflowsDollarValue <- liabilitiesTermData %>%
  filter(entity_name %in% LCRbanksTestList) %>%
  group_by(date, entity_name, attribute_name) %>% 
  summarise(total = sum(value_number)/1e9) %>% 
  ungroup() %>% 
  left_join(liabilitiesWaterfallLookup, by = c("attribute_name" = "value")) %>% 
  group_by(date, entity_name, Measure, name) %>% 
  summarise(total = sum(total)) %>%
  ungroup()

# Calculate off balance sheet liabilities from LCR form
# Spread them out over residual maturity - see below for assumptions 

outflowsOffBalanceSheet <- outflowsOffBalanceSheetRaw %>% 
  left_join(offBalanceSheetLookup, by = c("attribute_name" = "attributeName")) %>% 
  group_by(date, entity_name, Label, weighted) %>% 
  summarise(total = sum(value_number)/1e9) %>% 
  ungroup()

# This code is a little complex but it basically assumes all committed facility 
# flows are overnight and all derivatives outflows are evenly spread throughout 
# the first week. 

offBalanceSheetWeightedValues <- tidyr::crossing(date = unique(outflowsOffBalanceSheet$date), 
                                                 entity_name = unique(outflowsOffBalanceSheet$entity_name),
                                                 label = unique(outflowsOffBalanceSheet$Label), 
                                                 maturity = outflowsDollarValue$name) %>% 
  dplyr::mutate(days = case_when(maturity == "Overnight" ~ 1, 
                                 maturity == "2 to 7 days" ~ 6, 
                                 TRUE ~ 0)) %>%
  dplyr::filter(days != 0, 
                label == "Derivatives" | (label != "Derivatives" & maturity == "Overnight")) %>% 
  dplyr::group_by(date, entity_name, label) %>% 
  dplyr::mutate(dayShare = days/sum(days)) %>% 
  dplyr::ungroup() %>% 
  left_join(outflowsOffBalanceSheet %>% 
              filter(weighted == "Yes"), by = c("date", 
                                                "label" = "Label", 
                                                "entity_name")) %>% 
  mutate(total = total*dayShare) %>% 
  ungroup() %>% 
  select(date, entity_name, label, maturity, total) %>% 
  `colnames<-`(c("date", "entity_name", "Measure", "name", "total")) %>% 
  mutate(Measure = case_when(Measure != "Derivatives" ~ "Committed and contingent", 
                             TRUE ~ Measure)) %>% 
  group_by(date, entity_name, Measure, name) %>% 
  summarise(weightedAmount = sum(total)) %>% 
  ungroup() %>% 
  mutate(total = weightedAmount)

# This says by bank but it actually applies system-wide average
# outflow weights to each bank. Enduces homogeneity which 
# would be unfavourable for bank-level analysis


# By bank, category, residual maturity over time
LCRWeightedGrossOutflowByBank <- outflowsDollarValue %>% 
  mutate(Measure = case_when(Measure %in% c("Other - interest baring", 
                                            "Other - non-interest baring") ~ "Other", 
                             grepl("Unsecured Wholesale Non-financial corporate", Measure) ~ "NFCPSE",
                             TRUE ~ Measure)) %>% 
  left_join(LCROutflowsWeighted %>% 
              select(date, Measure, averageWeight), by = c("date", 
                                                           "Measure")) %>% 
  mutate(weightedAmount = total*averageWeight) %>% 
  bind_rows(offBalanceSheetWeightedValues) %>%
  filter(date >= as.Date("2018-01-01")) %>% 
  group_by(date, entity_name, Measure, name) %>% 
  summarise(unweightedOutflow = sum(total), 
            weightedOutflow = sum(weightedAmount)) %>% 
  ungroup() 

# By category, residual maturity over time
LCRWeightedGrossOutflow <- LCRWeightedGrossOutflowByBank %>% 
  group_by(date, Measure, name) %>% 
  summarise(unweightedOutflow = sum(unweightedOutflow, na.rm = TRUE), 
            weightedOutflow = sum(weightedOutflow, na.rm = TRUE)) %>% 
  ungroup() 

## Inflows -----------------------------------------------------------------

actualInflowWeights <- inflowComponentsFromLCR %>% 
  left_join(inflowLookupTable %>% 
              pivot_longer(-Item) %>% 
              drop_na(), by = c("attribute_name" = "value")) %>% 
  mutate(Item = case_when(Item %in% c("Retail", "SME") ~ "retail_SME",
                          Item %in% c("Central bank", "CLF", "Other", "ParentSupport") ~ "Other",
                          TRUE ~ Item)) %>% 
  group_by(date, entity_name, Item, name) %>% 
  summarise(total = sum(as.numeric(value_number))/1e9) %>%
  ungroup() %>% 
  pivot_wider(names_from = "name", values_from = total) %>% 
  replace(is.na(.), 0) %>% 
  drop_na() %>% 
  mutate(weight = Weighted/Unweighted, 
         type = "LCR") %>% 
  rename("period" = "date")

# Keep 
inflowWeighted <- inflowDataRaw %>% 
  filter(entity_name %in% LCRbanksTestList) %>%
  left_join(attrLookupInflows, by = c("attribute_name" = "value")) %>% 
  group_by(date, entity_name, Group, name) %>% 
  summarise(total = sum(value_number)/1e9) %>% 
  ungroup() %>% 
  left_join(actualInflowWeights %>% 
              select(period, entity_name, Item, weight), 
            by = c("date" = "period", 
                   "Group" = "Item", 
                   "entity_name")) %>% 
  mutate(weightedAmount = total*weight)

# Since we can't get inflows to line up well, I take the share 
# of each tenor within a category and apply it to the LCR number. 
# E.g. 50% of secured lending inflows are overnight. Therefore, overnight 
# inflows = secured lending reported in LCR * 50%. That way, my measure of 
# inflows is exactly equal to inflows reported in LCR - they are just split up 
# using tenor. 

inflowWeights <- inflowWeighted %>%
  group_by(date, entity_name, Group) %>% 
  mutate(share = weightedAmount/sum(weightedAmount)) %>% 
  ungroup() %>% 
  select(date, entity_name, Group, name, share)

onBalanceSheetInflows <- inflowWeights %>% 
  full_join(actualInflowWeights %>% 
              select(period, entity_name, Item, Weighted), by = c("date" = "period", 
                                                                  "entity_name", 
                                                                  "Group" = "Item")) %>% 
  filter(Group != "Derivatives") %>% 
  ungroup() %>% 
  mutate(value = Weighted*share) %>% 
  select(date, entity_name, Group, name, value)

inflowsDerivativesPossibilities <- tidyr::crossing(date = unique(onBalanceSheetInflows$date), 
                                                   entity_name = LCRbanksTestList,
                                                   Item = "Derivatives", 
                                                   maturity = unique(onBalanceSheetInflows$name)) %>% 
  dplyr::mutate(days = case_when(maturity == "Overnight" ~ 1, 
                                 maturity == "2 to 7 days" ~ 6, 
                                 TRUE ~ 0)) %>%
  dplyr::filter(days != 0) %>% 
  dplyr::group_by(date, entity_name, Item) %>% 
  dplyr::mutate(dayShare = days/sum(days)) %>% 
  dplyr::ungroup() %>% 
  left_join(actualInflowWeights %>% 
              filter(Item == "Derivatives"), by = c("date" = "period", 
                                                    "Item", 
                                                    "entity_name")) %>% 
  mutate(total = Weighted*dayShare) %>% 
  ungroup() %>% 
  select(date, entity_name, Item, maturity, total) %>% 
  `colnames<-`(c("date", "entity_name", "Group", "name", "value")) 

finalInflowsByBank <- onBalanceSheetInflows %>% 
  bind_rows(inflowsDerivativesPossibilities)

finalInflows <- finalInflowsByBank %>% 
  group_by(date, Group, name) %>% 
  summarise(value = sum(value, na.rm = TRUE)) %>% 
  ungroup() %>% 
  filter(value != 0)

## NCOs ----------------------------------------------------------------

netCashOutflowsByMaturity <- LCRWeightedGrossOutflow %>%
  group_by(date, name) %>% 
  summarise(weightedOutflow = sum(weightedOutflow)) %>%
  ungroup() %>%
  left_join(
    # collateral swaps not accounted for 
    finalInflows %>%
      group_by(date, name) %>% 
      summarise(weightedInflow = sum(value, na.rm = TRUE)) %>%
      ungroup() 
    
  ) %>% 
  mutate(NCOs = weightedOutflow - weightedInflow) 

# Graphs ------------------------------------------------------------------

## Graph 1 - Overnight NCOs ------------------------------------------------

NCOByMatWide <- netCashOutflowsByMaturity %>% 
  select(date, name, NCOs) %>% 
  pivot_wider(names_from = name, values_from = NCOs) %>% 
  janitor::adorn_totals(name = "totalNCO", where = "col") %>% 
  tibble() %>% 
  select(date, Overnight, totalNCO)

totalLiab <- totalLiabilitiesAndCapital %>% 
  group_by(date) %>% 
  summarise(totalLiabCap = sum(value_number)/1e9) %>% 
  ungroup()

info <- totalLiab %>% 
  left_join(NCOByMatWide, by = c("date")) %>% 
  mutate(NCOOvernightShare = Overnight/totalLiabCap*100)

overnightBS <- LCRWeightedGrossOutflow %>%
  filter(name == "Overnight") %>% 
  mutate(group = case_when(Measure %in% c("Retail Deposits") ~ "Retail at-call",
                           Measure %in% c("Unsecured Wholesale ADI",
                                          "Unsecured Wholesale Other",
                                          "Unsecured Wholesale SME",
                                          "NFCPSE") ~ "Wholesale at-call",
                           TRUE ~ "Other")) %>%
  group_by(date) %>% 
  summarise(total = sum(unweightedOutflow)) %>% 
  left_join(totalLiab) %>% 
  mutate(share = total/totalLiabCap*100) %>% 
  ungroup()

overnightNCOs <- NCOByMatWide %>% 
  select(date, Overnight, totalNCO) %>% 
  `colnames<-`(c("date", "overnightNCOs", "totalNCO"))

overnightFundingUnweighted <- overnightBS %>% 
  select(date, total, share) %>% 
  `colnames<-`(c("date", "overnightFunding", "overnightFundingShare"))

creditLines <- LCRWeightedGrossOutflow %>% 
  filter(Measure == "Committed and contingent") %>% 
  group_by(date) %>% 
  summarise(creditLines = sum(weightedOutflow)) %>% 
  ungroup()

overnightNCOShare <- info %>% 
  select(date, NCOOvernightShare) %>% 
  `colnames<-`(c("date", "overnightNCOsShare"))


overnightBS <- LCRWeightedGrossOutflow %>%
  filter(name == "Overnight") %>% 
  mutate(group = case_when(Measure %in% c("Retail Deposits") ~ "Retail at-call",
                           Measure %in% c("Unsecured Wholesale ADI",
                                          "Unsecured Wholesale Other",
                                          "Unsecured Wholesale SME",
                                          "NFCPSE") ~ "Wholesale at-call",
                           Measure %in% c("Committed and contingent") ~ Measure, 
                           TRUE ~ "Other")) %>%
  group_by(date, group) %>% 
  summarise(total = sum(unweightedOutflow)) %>% 
  left_join(totalLiab) %>% 
  mutate(share = total/totalLiabCap*100) %>% 
  ungroup()

left_join(overnightNCOs, overnightFundingUnweighted) %>% 
  left_join(creditLines) %>% 
  left_join(overnightNCOShare) %>% 
  arrange(date) %>% 
  write.csv(x = ., 
            file = paste0(projectDir, "overnightNCOsAndFunding.csv"))
